let currentRecord = null;

function showMessage(message, isSuccess) {
    const messageDiv = document.getElementById('message');
    messageDiv.textContent = message;
    messageDiv.className = 'message ' + (isSuccess ? 'success' : 'error');
    messageDiv.style.display = 'block';
    
    setTimeout(() => {
        messageDiv.style.display = 'none';
    }, 5000);
}

function clearForm() {
    document.getElementById('refno').value = '';
    document.getElementById('sdate').value = '';
    document.getElementById('entdate').value = '';
    document.getElementById('sal_ex').value = '';
    document.getElementById('town').value = '';
    document.getElementById('townSelect').value = '';
    document.getElementById('area').value = '';
    document.getElementById('areaSelect').value = '';
    document.getElementById('enable').value = '1';
    currentRecord = null;
    
    // Remove selection from all records
    document.querySelectorAll('.record-item').forEach(item => {
        item.classList.remove('selected');
    });
}

function loadRecords() {
    fetch('backend.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: 'action=list'
    })
    .then(response => {
        if (!response.ok) {
            throw new Error('Network response was not ok');
        }
        return response.json();
    })
    .then(data => {
        console.log('Load records response:', data);
        if (data.success) {
            displayRecords(data.records);
        } else {
            showMessage(data.message || 'Error loading records', false);
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showMessage('Error loading records: ' + error.message, false);
    });
}

function displayRecords(records) {
    const container = document.getElementById('recordsContainer');
    container.innerHTML = '';
    
    if (!records || records.length === 0) {
        container.innerHTML = '<p style="text-align:center; color:#999; padding:20px;">No records found</p>';
        return;
    }
    
    records.forEach(record => {
        const recordDiv = document.createElement('div');
        recordDiv.className = 'record-item';
        recordDiv.innerHTML = `
            <strong>${record.refno}</strong> - ${record.Town} - ${record.Area || 'N/A'} - ${record.sdate}
            <br><small>Sales: ${record.sal_ex} | Enable: ${record.Enable == 1 ? 'Yes' : 'No'}</small>
        `;
        
        recordDiv.onclick = function() {
            // Remove selected class from all items
            document.querySelectorAll('.record-item').forEach(item => {
                item.classList.remove('selected');
            });
            // Add selected class to clicked item
            this.classList.add('selected');
            loadRecord(record.refno);
        };
        
        container.appendChild(recordDiv);
    });
}

function loadRecord(refno) {
    fetch('backend.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: `action=load&refno=${encodeURIComponent(refno)}`
    })
    .then(response => response.json())
    .then(data => {
        console.log('Load record response:', data);
        if (data.success) {
            const record = data.data;
            document.getElementById('refno').value = record.refno;
            document.getElementById('sdate').value = record.sdate;
            document.getElementById('entdate').value = record.entdate;
            document.getElementById('sal_ex').value = record.sal_ex;
            document.getElementById('town').value = record.Town;
            document.getElementById('townSelect').value = record.Town;
            document.getElementById('area').value = record.Area || '';
            document.getElementById('areaSelect').value = record.Area || '';
            document.getElementById('enable').value = record.Enable;
            currentRecord = record.refno;
        } else {
            showMessage(data.message || 'Error loading record', false);
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showMessage('Error loading record: ' + error.message, false);
    });
}

function newRecord() {
    fetch('backend.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: 'action=new'
    })
    .then(response => response.json())
    .then(data => {
        console.log('New record response:', data);
        if (data.success) {
            clearForm();
            document.getElementById('refno').value = data.refno;
            showMessage(data.message, true);
            // Set today's date
            const today = new Date().toISOString().split('T')[0];
            document.getElementById('sdate').value = today;
            document.getElementById('entdate').value = today;
        } else {
            showMessage(data.message || 'Error generating new reference', false);
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showMessage('Error generating new reference: ' + error.message, false);
    });
}

function saveRecord() {
    const formData = new FormData();
    formData.append('action', 'save');
    formData.append('refno', document.getElementById('refno').value);
    formData.append('sdate', document.getElementById('sdate').value);
    formData.append('entdate', document.getElementById('entdate').value);
    formData.append('sal_ex', document.getElementById('sal_ex').value);
    formData.append('town', document.getElementById('town').value);
    formData.append('area', document.getElementById('area').value);
    formData.append('enable', document.getElementById('enable').value);

    fetch('backend.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        console.log('Save record response:', data);
        showMessage(data.message || 'Operation completed', data.success);
        if (data.success) {
            loadRecords();
            clearForm();
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showMessage('Error saving record: ' + error.message, false);
    });
}

function updateRecord() {
    if (!currentRecord) {
        showMessage('Please select a record to update', false);
        return;
    }

    const formData = new FormData();
    formData.append('action', 'update');
    formData.append('refno', document.getElementById('refno').value);
    formData.append('sdate', document.getElementById('sdate').value);
    formData.append('entdate', document.getElementById('entdate').value);
    formData.append('sal_ex', document.getElementById('sal_ex').value);
    formData.append('town', document.getElementById('town').value);
    formData.append('area', document.getElementById('area').value);
    formData.append('enable', document.getElementById('enable').value);

    fetch('backend.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        console.log('Update record response:', data);
        showMessage(data.message || 'Operation completed', data.success);
        if (data.success) {
            loadRecords();
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showMessage('Error updating record: ' + error.message, false);
    });
}

function deleteRecord() {
    if (!currentRecord) {
        showMessage('Please select a record to delete', false);
        return;
    }

    if (!confirm('Are you sure you want to delete this record?')) {
        return;
    }

    fetch('backend.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: `action=delete&refno=${encodeURIComponent(currentRecord)}`
    })
    .then(response => response.json())
    .then(data => {
        console.log('Delete record response:', data);
        showMessage(data.message || 'Operation completed', data.success);
        if (data.success) {
            loadRecords();
            clearForm();
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showMessage('Error deleting record: ' + error.message, false);
    });
}



// Initialize when page loads
document.addEventListener('DOMContentLoaded', function() {
    console.log('Page loaded, initializing...');
    loadRecords();
    
    // Set today's date as default for date fields
    const today = new Date().toISOString().split('T')[0];
    document.getElementById('sdate').value = today;
    document.getElementById('entdate').value = today;
    
    console.log('Initialization complete');
});
