<?php
session_start();
date_default_timezone_set('Asia/Colombo');

// Basic session check
if (!isset($_SESSION["CURRENT_USER"]) || $_SESSION["CURRENT_USER"] == "") {
    echo "Please Login Again !!!";
    exit();
}

// Database connection
require_once("security/connection_sql.php");

// Ensure PDO throws exceptions
if (isset($conn) && $conn instanceof PDO) {
    $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} else {
    die("Database connection not available.");
}

// Fetch distinct Sales Executives
$reps = [];
try {
    $stmt = $conn->query("SELECT DISTINCT ref_no1 FROM vendor_visit WHERE ref_no1 IS NOT NULL AND TRIM(ref_no1) <> '' ORDER BY ref_no1");
    $reps = $stmt->fetchAll(PDO::FETCH_COLUMN);
} catch (Throwable $e) {
    $reps = [];
}

// Get default date (today)
$today = date('Y-m-d');

// Process form submission
$selected_date = isset($_POST['date']) ? $_POST['date'] : $today;
$selected_rep = isset($_POST['sal_ex']) ? $_POST['sal_ex'] : '';

// Fetch route data if form submitted
$route_data = [];
$total_visits = 0;
$total_collection = 0;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $sql = "SELECT 
                sdate1, cus_name, ref_no, collect, latitude, longitude, 
                flag, collection_time
            FROM vendor_visit 
            WHERE DATE(sdate1) = :selected_date 
            AND latitude IS NOT NULL 
            AND longitude IS NOT NULL";

        $params = [':selected_date' => $selected_date];
        
        if (!empty($selected_rep)) {
            $sql .= " AND ref_no1 = :sal_ex";
            $params[':sal_ex'] = $selected_rep;
        }
        
        $sql .= " ORDER BY COALESCE(collection_time, sdate1) ASC, sdate1 ASC";

        
        $stmt = $conn->prepare($sql);
        $stmt->execute($params);
        $route_data = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Calculate summary stats
        $total_visits = count($route_data);
        foreach ($route_data as $visit) {
            //$total_collection += floatval($visit['collect']);
            // Only add to total if collect is numeric
            if (is_numeric($visit['collect'])) {
                $total_collection += floatval($visit['collect']);
            }

        }
        
        // Add pre-formatted Sri Lanka time for JS popups and timeline
        foreach ($route_data as &$visit) {
            // Format visit time
            $visit_datetime = strtotime($visit['sdate1']);
            $visit['visit_time'] = date('h:i A', $visit_datetime);
            
            // Format collection save time if available
            if (isset($visit['collection_time']) && !empty($visit['collection_time'])) {
                $collection_ts = strtotime($visit['collection_time']);
                $visit['formatted_collection_time'] = date('h:i A', $collection_ts);
            } else {
                $visit['formatted_collection_time'] = null;
            }
        }
        unset($visit);
        
    } catch (Throwable $e) {
        $error = "Error fetching route data: " . $e->getMessage();
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>ME Route Tracking</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <!-- Leaflet CSS -->
    <link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.4/dist/leaflet.css" integrity="sha256-p4NxAoJBhIIN+hmNHrzRCf9tD/miZyoHS5obTRR9BMY=" crossorigin=""/>
    <style>
        body { background-color: #f8f9fa; }
        .map-header {
            background: linear-gradient(135deg, #075E54 0%, #128C7E 100%);
            color: white;
            padding: 2rem 0;
            margin-bottom: 2rem;
        }
        #map { 
            height: 600px; 
            width: 100%;
            border-radius: 8px;
            box-shadow: 0 0 15px rgba(0,0,0,0.1);
        }
        .summary-card { 
            border: none; 
            border-radius: 12px; 
            box-shadow: 0 3px 10px rgba(0,0,0,0.08); 
            margin-bottom: 20px;
        }
        .legend {
            padding: 10px;
            background-color: white;
            border-radius: 5px;
            box-shadow: 0 0 10px rgba(0,0,0,0.1);
        }
        .legend i {
            width: 18px;
            height: 18px;
            float: left;
            margin-right: 8px;
            opacity: 0.7;
        }
        .visit-list {
            height: 600px;
            overflow-y: auto;
        }
        .visit-item {
            border-left: 4px solid #128C7E;
            margin-bottom: 10px;
            transition: all 0.2s;
        }
        .visit-item:hover {
            background-color: #f1f8ff;
            cursor: pointer;
            transform: translateX(5px);
        }
        .visit-time {
            font-weight: bold;
            color: #075E54;
        }
        .visit-marker {
            font-size: 11px;
            color: #333;
            font-weight: bold;
        }
    </style>
</head>
<body>

<div class="map-header">
    <div class="container">
        <div class="row align-items-center">
            <div class="col-md-8">
                <a href="collection_assign_report.php" class="btn btn-light" id="backButton">
                        <i class="fas fa-arrow-left me-1"></i> Back 
                 </a>
                <h1><i class="fas fa-route me-2"></i>ME Route Tracking</h1>
                <p class="lead mb-0">Visualize Mobile Executive movement and collection activities</p>
            </div>
        </div>
    </div>
</div>

<div class="container">
    <!-- Filter Form -->
    <div class="card mb-4">
        <div class="card-header bg-primary text-white">
            <h5 class="mb-0"><i class="fas fa-filter me-2"></i>Select Route Details</h5>
        </div>
        <div class="card-body">
            <form method="POST" class="row g-3">
                <div class="col-md-5">
                    <label for="sal_ex" class="form-label">Sales Executive</label>
                    <select class="form-select" id="sal_ex" name="sal_ex">
                        <option value="">All Executives</option>
                        <?php foreach ($reps as $rep): ?>
                            <option value="<?= htmlspecialchars($rep) ?>" <?= ($selected_rep === $rep) ? 'selected' : '' ?>>
                                <?= htmlspecialchars($rep) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="col-md-5">
                    <label for="date" class="form-label">Date</label>
                    <input type="date" class="form-control" id="date" name="date" value="<?= htmlspecialchars($selected_date) ?>" required>
                </div>
                <div class="col-md-2 d-flex align-items-end">
                    <button type="submit" class="btn btn-success w-100"><i class="fas fa-map-marked-alt me-1"></i>View Route</button>
                </div>
            </form>
        </div>
    </div>

    <?php if ($_SERVER['REQUEST_METHOD'] === 'POST'): ?>
        <?php if (empty($route_data)): ?>
            <div class="alert alert-info text-center">
                <i class="fas fa-info-circle me-2"></i>No location data found for the selected criteria.
            </div>
        <?php else: ?>
            <!-- Summary Cards -->
            <div class="row mb-4">
                <div class="col-md-4 mb-3">
                    <div class="card summary-card text-center bg-light">
                        <div class="card-body">
                            <i class="fas fa-user fa-2x text-primary mb-2"></i>
                            <h4 class="card-title"><?= htmlspecialchars(!empty($selected_rep) ? $selected_rep : 'All Executives') ?></h4>
                            <p class="card-text text-muted">Sales Executive</p>
                        </div>
                    </div>
                </div>
                <div class="col-md-4 mb-3">
                    <div class="card summary-card text-center bg-light">
                        <div class="card-body">
                            <i class="fas fa-map-marker-alt fa-2x text-danger mb-2"></i>
                            <h4 class="card-title"><?= $total_visits ?></h4>
                            <p class="card-text text-muted">Total Visits</p>
                        </div>
                    </div>
                </div>
                <div class="col-md-4 mb-3">
                    <div class="card summary-card text-center bg-light">
                        <div class="card-body">
                            <i class="fas fa-money-bill-wave fa-2x text-success mb-2"></i>
                            <h4 class="card-title">LKR <?= number_format($total_collection, 2) ?></h4>
                            <p class="card-text text-muted">Total Collection</p>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Map and Visit List -->
            <div class="row">
                <div class="col-md-8">
                    <div class="card">
                        <div class="card-header bg-success text-white">
                            <h5 class="mb-0"><i class="fas fa-map me-2"></i>Route Map</h5>
                        </div>
                        <div class="card-body p-0">
                            <div id="map"></div>
                        </div>
                    </div>
                </div>
                <div class="col-md-4">
                    <div class="card">
                        <div class="card-header bg-info text-white">
                            <h5 class="mb-0"><i class="fas fa-list me-2"></i>Visits Timeline</h5>
                        </div>
                        <div class="card-body p-0">
                            <div class="visit-list p-2">
                                <?php foreach ($route_data as $index => $visit): ?>
                                    <div class="card visit-item" data-index="<?= $index ?>">
                                        <div class="card-body">
                                            <div class="visit-time">
                                                <i class="far fa-clock me-1"></i>
                                                <?php 
                                                    // Fix: Extract and format time from datetime, only show if not midnight
                                                    $visit_datetime = strtotime($visit['sdate1']);
                                                    $time_str = date('h:i A', $visit_datetime);
                                                    
                                                    // Only display time if it's not 12:00 AM (which indicates no time was set)
                                                    if ($time_str !== '12:00 AM') {
                                                        echo $time_str;
                                                    } else {
                                                        echo date('Y-m-d', $visit_datetime); // Just show date if time is not available
                                                    }
                                                ?>
                                                <span class="badge bg-<?= $visit['flag'] === 'checq' ? 'warning' : 'info' ?> float-end">
                                                    <?= $visit['flag'] === 'checq' ? 'Cheque' : 'Invoice' ?>
                                                </span>
                                            </div>
                                            <div class="mt-1">
                                                <strong><?= htmlspecialchars($visit['cus_name']) ?></strong>
                                            </div>
                                            <div class="text-muted small">
                                                <?= htmlspecialchars($visit['ref_no']) ?>
                                            </div>
                                            <?php if(isset($visit['formatted_collection_time']) && $visit['formatted_collection_time']): ?>
                                            <div class="text-muted small">
                                                <i class="fas fa-save me-1"></i> Saved at: <?= htmlspecialchars($visit['formatted_collection_time']) ?>
                                            </div>
                                            <?php endif; ?>
                                            <div class="mt-2">
                                                <span class="badge bg-success">LKR <?= number_format($visit['collect'], 2) ?></span>
                                            </div>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        <?php endif; ?>
    <?php endif; ?>
</div>

<!-- Leaflet JS -->
<script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js" integrity="sha256-20nQCchB9co0qIjJZRGuk2/Z9VM+kNiyxNV1lvTlZBo=" crossorigin=""></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>

<?php if (!empty($route_data)): ?>
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Initialize the map centered on Sri Lanka
    var map = L.map('map').setView([7.8731, 80.7718], 8);
    
    // Add OpenStreetMap tile layer
    L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
        attribution: '&copy; <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors'
    }).addTo(map);
    
    // Route data from PHP
    var routeData = <?= json_encode($route_data) ?>;
    var markers = [];
    var latlngs = [];
    
    // Add markers for each visit
    routeData.forEach(function(visit, index) {
        if (visit.latitude && visit.longitude) {
            var lat = parseFloat(visit.latitude);
            var lng = parseFloat(visit.longitude);

            if (isNaN(lat) || isNaN(lng) || lat === 0 || lng === 0) {
                console.log("Invalid coordinates for visit:", visit);
                return;
            }
            
            latlngs.push([lat, lng]);
            
            // Create custom icon with number
            var markerIcon = L.divIcon({
                html: '<div style="background-color: ' + 
                    (visit.flag === 'checq' ? '#ffc107' : '#17a2b8') + 
                    '; width: 30px; height: 30px; border-radius: 15px; display: flex; justify-content: center; align-items: center; border: 2px solid white; box-shadow: 0 0 5px rgba(0,0,0,0.3);">' + 
                    '<span class="visit-marker">' + (index + 1) + '</span></div>',
                className: '',
                iconSize: [30, 30]
            });
            
            // Format the time properly for popup
            var visitDateTime = new Date(visit.sdate1);
            var formattedTime = '';
            
            
           
            
            // Get collection time if available
            var collectionTimeInfo = "";
            if (visit.formatted_collection_time) {
                collectionTimeInfo = "Collection Saved: " + visit.formatted_collection_time + "<br>";
            }
            
            // Create marker
            var marker = L.marker([lat, lng], {icon: markerIcon})
                .bindPopup(
                    "<strong>" + visit.cus_name + "</strong><br>" +
                    collectionTimeInfo +
                    "Amount: LKR " + parseFloat(visit.collect).toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2}) + "<br>" +
                    "Ref: " + visit.ref_no + "<br>" +
                    "Type: " + (visit.flag === 'checq' ? 'Cheque' : 'Invoice')
                );
            
            markers.push(marker);
            marker.addTo(map);
        }
    });
    
    // Draw the route line if we have 2 or more points
    if (latlngs.length >= 2) {
        var routeLine = L.polyline(latlngs, {
            color: '#075E54',
            weight: 3,
            opacity: 0.7,
            dashArray: '10, 10',
            lineJoin: 'round'
        }).addTo(map);
        
        // Fit the map to show all markers
        map.fitBounds(routeLine.getBounds(), {
            padding: [50, 50]
        });
    } else if (latlngs.length === 1) {
        // If only one point, zoom to it
        map.setView(latlngs[0], 13);
    }
    
    // Add a legend
    var legend = L.control({position: 'bottomright'});
    legend.onAdd = function(map) {
        var div = L.DomUtil.create('div', 'legend');
        div.innerHTML = 
            '<h6>Legend</h6>' +
            '<div><i style="background: #ffc107;"></i> Cheque Collection</div>' +
            '<div><i style="background: #17a2b8;"></i> Invoice Collection</div>' +
            '<div><i style="background: #075E54; height: 3px;"></i> Travel Route</div>';
        return div;
    };
    legend.addTo(map);
    
    // Handle clicking on visit items in the timeline
    document.querySelectorAll('.visit-item').forEach(function(item) {
        item.addEventListener('click', function() {
            var index = parseInt(this.getAttribute('data-index'));
            if (markers[index]) {
                markers[index].openPopup();
                map.setView(markers[index].getLatLng(), 15);
            }
        });
    });
});
</script>
<?php endif; ?>

</body>
</html>
