<?php
session_start();
date_default_timezone_set('Asia/Colombo');

// Basic session check
if (!isset($_SESSION["CURRENT_USER"]) || $_SESSION["CURRENT_USER"] == "") {
    echo "Please Login Again !!!";
    exit();
}

// Database connection
require_once("security/connection_sql.php");

// Ensure PDO throws exceptions
if (isset($conn) && $conn instanceof PDO) {
    $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} else {
    die("Database connection not available.");
}

// Fetch distinct Sales Executives
$reps = [];
try {
    $stmt = $conn->query("SELECT DISTINCT ref_no1 FROM vendor_visit WHERE ref_no1 IS NOT NULL AND TRIM(ref_no1) <> '' ORDER BY ref_no1");
    $reps = $stmt->fetchAll(PDO::FETCH_COLUMN);
} catch (Throwable $e) {
    $reps = [];
}



// Process form submission
$filters = [
    'sal_ex' => isset($_POST['sal_ex']) ? $_POST['sal_ex'] : '',
    'start_date' => isset($_POST['start_date']) ? $_POST['start_date'] : date('Y-m-01'), // First day of current month
    'end_date' => isset($_POST['end_date']) ? $_POST['end_date'] : date('Y-m-d')  // Today
];

// Analysis data
$analysis_data = [];
$total_visits = 0;
$total_collection = 0;
$daily_stats = [];
$rep_stats = [];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        // Get daily visit counts and collection amounts
        $sql = "SELECT 
                DATE(sdate1) as visit_date,
                ref_no1 as sales_exec,
                COUNT(*) as visit_count,
                SUM(collect) as daily_collection,
                SUM(CASE WHEN flag = 'checq' THEN 1 ELSE 0 END) as cheque_count,
                SUM(CASE WHEN flag = 'inv' THEN 1 ELSE 0 END) as invoice_count
            FROM vendor_visit 
            WHERE DATE(sdate1) BETWEEN :start_date AND :end_date
            AND latitude IS NOT NULL AND longitude IS NOT NULL";
        
        $params = [
            ':start_date' => $filters['start_date'],
            ':end_date' => $filters['end_date']
        ];
        
        if (!empty($filters['sal_ex'])) {
            $sql .= " AND ref_no1 = :sal_ex";
            $params[':sal_ex'] = $filters['sal_ex'];
        }
        
        $sql .= " GROUP BY DATE(sdate1), ref_no1 ORDER BY DATE(sdate1)";
        
        $stmt = $conn->prepare($sql);
        $stmt->execute($params);
        $analysis_data = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Process the analysis data
        foreach ($analysis_data as $day) {
            $date = $day['visit_date'];
            $exec = $day['sales_exec'];
            
            // Build daily stats
            if (!isset($daily_stats[$date])) {
                $daily_stats[$date] = [
                    'date' => $date,
                    'total_visits' => 0,
                    'total_collection' => 0,
                    'executives' => [],
                    'cheque_count' => 0,
                    'invoice_count' => 0
                ];
            }
            
            $daily_stats[$date]['total_visits'] += $day['visit_count'];
            $daily_stats[$date]['total_collection'] += $day['daily_collection'];
            $daily_stats[$date]['cheque_count'] += $day['cheque_count'];
            $daily_stats[$date]['invoice_count'] += $day['invoice_count'];
            $daily_stats[$date]['executives'][] = $exec;
            
            // Build rep stats
            if (!isset($rep_stats[$exec])) {
                $rep_stats[$exec] = [
                    'name' => $exec,
                    'total_visits' => 0,
                    'total_collection' => 0,
                    'days_active' => 0,
                    'cheque_count' => 0,
                    'invoice_count' => 0
                ];
            }
            
            $rep_stats[$exec]['total_visits'] += $day['visit_count'];
            $rep_stats[$exec]['total_collection'] += $day['daily_collection'];
            $rep_stats[$exec]['days_active']++;
            $rep_stats[$exec]['cheque_count'] += $day['cheque_count'];
            $rep_stats[$exec]['invoice_count'] += $day['invoice_count'];
            
            // Update overall totals
            $total_visits += $day['visit_count'];
            $total_collection += $day['daily_collection'];
        }
        
        // Sort daily stats by date
        ksort($daily_stats);
        
        // Sort rep stats by total collection (highest first)
        uasort($rep_stats, function($a, $b) {
            if ($a['total_collection'] == $b['total_collection']) {
                return 0;
            }
            return ($a['total_collection'] < $b['total_collection']) ? 1 : -1;
        });
        
    } catch (Throwable $e) {
        $error = "Error analyzing route data: " . $e->getMessage();
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>ME Route Analysis</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <style>
        body { background-color: #f8f9fa; }
        .analysis-header {
            background: linear-gradient(135deg, #6f42c1 0%, #007bff 100%);
            color: white;
            padding: 2rem 0;
            margin-bottom: 2rem;
        }
        .summary-card { 
            border: none; 
            border-radius: 12px; 
            box-shadow: 0 3px 10px rgba(0,0,0,0.08); 
            margin-bottom: 20px;
            transition: transform 0.3s;
        }
        .summary-card:hover {
            transform: translateY(-5px);
        }
        .chart-container {
            position: relative;
            height: 300px;
            width: 100%;
        }
        .rep-performance {
            border-left: 4px solid #007bff;
            padding-left: 15px;
            margin-bottom: 15px;
        }
        .heat-cell {
            width: 30px;
            height: 30px;
            margin: 2px;
            border-radius: 4px;
            display: inline-block;
            position: relative;
        }
        .heat-cell:hover::after {
            content: attr(data-tooltip);
            position: absolute;
            bottom: 100%;
            left: 50%;
            transform: translateX(-50%);
            background: #333;
            color: white;
            padding: 5px 10px;
            border-radius: 3px;
            font-size: 12px;
            white-space: nowrap;
            z-index: 10;
        }
        .heatmap-container {
            text-align: center;
            overflow-x: auto;
            white-space: nowrap;
            padding: 15px 5px;
        }
    </style>
</head>
<body>

<div class="analysis-header">
    <div class="container">
        <div class="row align-items-center">
            <div class="col-md-8">
                <a href="collection_assign_report.php" class="btn btn-light" id="backButton">
                        <i class="fas fa-arrow-left me-1"></i> Back 
                 </a>
                <h1><i class="fas fa-chart-line me-2"></i>ME Route Analysis</h1>
                <p class="lead mb-0">Comprehensive analysis of Mobile Executive activities</p>
            </div>
            <div class="col-md-4 text-end">
                <button onclick="window.print()" class="btn btn-light btn-sm">
                    <i class="fas fa-print me-1"></i>Print Analysis
                </button>
            </div>
        </div>
    </div>
</div>

<div class="container">
    <!-- Filter Form -->
    <div class="card mb-4">
        <div class="card-header bg-primary text-white">
            <h5 class="mb-0"><i class="fas fa-filter me-2"></i>Filter Analysis</h5>
        </div>
        <div class="card-body">
            <form method="POST" class="row g-3">
                <div class="col-md-4">
                    <label for="sal_ex" class="form-label">Sales Executive</label>
                    <select class="form-select" id="sal_ex" name="sal_ex">
                        <option value="">All Executives</option>
                        <?php foreach ($reps as $rep): ?>
                            <option value="<?= htmlspecialchars($rep) ?>" <?= ($filters['sal_ex'] === $rep) ? 'selected' : '' ?>>
                                <?= htmlspecialchars($rep) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="col-md-3">
                    <label for="start_date" class="form-label">Start Date</label>
                    <input type="date" class="form-control" id="start_date" name="start_date" value="<?= htmlspecialchars($filters['start_date']) ?>" required>
                </div>
                <div class="col-md-3">
                    <label for="end_date" class="form-label">End Date</label>
                    <input type="date" class="form-control" id="end_date" name="end_date" value="<?= htmlspecialchars($filters['end_date']) ?>" required>
                </div>
                <div class="col-md-2 d-flex align-items-end">
                    <button type="submit" class="btn btn-success w-100"><i class="fas fa-chart-bar me-1"></i>Analyze</button>
                </div>
            </form>
        </div>
    </div>

    <?php if (isset($error)): ?>
        <div class="alert alert-danger"><?= htmlspecialchars($error) ?></div>
    <?php endif; ?>

    <?php if ($_SERVER['REQUEST_METHOD'] === 'POST' && !empty($analysis_data)): ?>
        <!-- Summary Cards -->
        <div class="row mb-4">
            <div class="col-md-3 mb-3">
                <div class="card summary-card text-center bg-light">
                    <div class="card-body">
                        <i class="fas fa-calendar-alt fa-2x text-primary mb-2"></i>
                        <h4 class="card-title"><?= count($daily_stats) ?></h4>
                        <p class="card-text text-muted">Active Days</p>
                    </div>
                </div>
            </div>
            <div class="col-md-3 mb-3">
                <div class="card summary-card text-center bg-light">
                    <div class="card-body">
                        <i class="fas fa-user-tie fa-2x text-info mb-2"></i>
                        <h4 class="card-title"><?= count($rep_stats) ?></h4>
                        <p class="card-text text-muted">Active Executives</p>
                    </div>
                </div>
            </div>
            <div class="col-md-3 mb-3">
                <div class="card summary-card text-center bg-light">
                    <div class="card-body">
                        <i class="fas fa-map-marker-alt fa-2x text-danger mb-2"></i>
                        <h4 class="card-title"><?= $total_visits ?></h4>
                        <p class="card-text text-muted">Total Visits</p>
                    </div>
                </div>
            </div>
            <div class="col-md-3 mb-3">
                <div class="card summary-card text-center bg-light">
                    <div class="card-body">
                        <i class="fas fa-money-bill-wave fa-2x text-success mb-2"></i>
                        <h4 class="card-title">LKR <?= number_format($total_collection, 2) ?></h4>
                        <p class="card-text text-muted">Total Collection</p>
                    </div>
                </div>
            </div>
        </div>

        <!-- Charts -->
        <div class="row mb-4">
            <div class="col-md-6 mb-4">
                <div class="card">
                    <div class="card-header bg-primary text-white">
                        <h5 class="mb-0"><i class="fas fa-chart-bar me-2"></i>Daily Visits</h5>
                    </div>
                    <div class="card-body">
                        <div class="chart-container">
                            <canvas id="visitsChart"></canvas>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-md-6 mb-4">
                <div class="card">
                    <div class="card-header bg-success text-white">
                        <h5 class="mb-0"><i class="fas fa-chart-line me-2"></i>Daily Collections</h5>
                    </div>
                    <div class="card-body">
                        <div class="chart-container">
                            <canvas id="collectionsChart"></canvas>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        

        <!-- Activity Heatmap -->
        <?php if (count($daily_stats) > 0): ?>
            <div class="card mb-4">
                <div class="card-header bg-info text-white">
                    <h5 class="mb-0"><i class="fas fa-th me-2"></i>Activity Heatmap</h5>
                </div>
                <div class="card-body p-0">
                    <div class="heatmap-container">
                        <?php 
                        // Find the max value for scaling
                        $max_visits = 0;
                        foreach ($daily_stats as $day) {
                            $max_visits = max($max_visits, $day['total_visits']);
                        }
                        
                        foreach ($daily_stats as $day): 
                            // Calculate intensity (0-100%)
                            $intensity = $max_visits > 0 ? ($day['total_visits'] / $max_visits) * 100 : 0;
                            $color = "rgba(40, 167, 69, " . ($intensity/100) . ")";
                            $date_formatted = date('M d', strtotime($day['date']));
                            $tooltip = $date_formatted . ": " . $day['total_visits'] . " visits, LKR " . 
                                      number_format($day['total_collection'], 2);
                        ?>
                            <div class="heat-cell" 
                                 style="background-color: <?= $color ?>;" 
                                 data-tooltip="<?= $tooltip ?>"></div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>
        <?php endif; ?>

        <!-- Executive Performance -->
        <?php if (count($rep_stats) > 0): ?>
            <div class="card mb-4">
                <div class="card-header bg-warning text-dark">
                    <h5 class="mb-0"><i class="fas fa-medal me-2"></i>Executive Performance</h5>
                </div>
                <div class="card-body">
                    <div class="row">
                        <?php foreach ($rep_stats as $rep): ?>
                            <div class="col-md-6 mb-3">
                                <div class="rep-performance">
                                    <h5><?= htmlspecialchars($rep['name']) ?></h5>
                                    <div class="d-flex justify-content-between">
                                        <div>
                                            <p class="mb-1">Total Visits: <strong><?= $rep['total_visits'] ?></strong></p>
                                            <p class="mb-1">Days Active: <strong><?= $rep['days_active'] ?></strong></p>
                                            <p class="mb-1">Avg. Visits/Day: <strong><?= round($rep['total_visits'] / $rep['days_active'], 1) ?></strong></p>
                                        </div>
                                        <div>
                                            <p class="mb-1">Cheque Collections: <strong><?= $rep['cheque_count'] ?></strong></p>
                                            <p class="mb-1">Invoice Collections: <strong><?= $rep['invoice_count'] ?></strong></p>
                                            <p class="mb-1">Total Collected: <strong>LKR <?= number_format($rep['total_collection'], 2) ?></strong></p>
                                        </div>
                                    </div>
                                    <div class="progress mt-2">
                                        <div class="progress-bar" role="progressbar" 
                                             style="width: <?= ($rep['total_collection'] / $total_collection) * 100 ?>%;" 
                                             aria-valuenow="<?= ($rep['total_collection'] / $total_collection) * 100 ?>" 
                                             aria-valuemin="0" 
                                             aria-valuemax="100">
                                             <?= round(($rep['total_collection'] / $total_collection) * 100, 1) ?>%
                                        </div>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>
        <?php endif; ?>

        <!-- Daily Details -->
        <div class="card">
            <div class="card-header bg-secondary text-white">
                <h5 class="mb-0"><i class="fas fa-list-alt me-2"></i>Daily Details</h5>
            </div>
            <div class="card-body p-0">
                <div class="table-responsive">
                    <table class="table table-striped table-hover mb-0">
                        <thead>
                            <tr>
                                <th>Date</th>
                                <th>Sales Executive</th>
                                <th class="text-center">Visits</th>
                                <th class="text-center">Cheques</th>
                                <th class="text-center">Invoices</th>
                                <th class="text-end">Collection</th>
                                <th>Action</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($analysis_data as $row): ?>
                                <tr>
                                    <td><?= date('Y-m-d (D)', strtotime($row['visit_date'])) ?></td>
                                    <td><?= htmlspecialchars($row['sales_exec']) ?></td>
                                    <td class="text-center"><?= $row['visit_count'] ?></td>
                                    <td class="text-center"><?= $row['cheque_count'] ?></td>
                                    <td class="text-center"><?= $row['invoice_count'] ?></td>
                                    <td class="text-end">LKR <?= number_format($row['daily_collection'], 2) ?></td>
                                    <td>
                                        <a href="route_tracking.php?date=<?= $row['visit_date'] ?>&sal_ex=<?= urlencode($row['sales_exec']) ?>" 
                                           class="btn btn-sm btn-primary">
                                            <i class="fas fa-map-marked-alt"></i> View Route
                                        </a>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>

        <script>
            // Charts initialization
            document.addEventListener('DOMContentLoaded', function() {
                // Prepare data for charts
                var dailyStats = <?= json_encode(array_values($daily_stats)) ?>;
                var dates = dailyStats.map(day => day.date);
                var visits = dailyStats.map(day => day.total_visits);
                var collections = dailyStats.map(day => day.total_collection);
                
                // Format dates for display
                var formattedDates = dates.map(date => {
                    var d = new Date(date);
                    return d.toLocaleDateString('en-US', {month: 'short', day: 'numeric'});
                });
                
                // Visits Chart
                var visitsCtx = document.getElementById('visitsChart').getContext('2d');
                new Chart(visitsCtx, {
                    type: 'bar',
                    data: {
                        labels: formattedDates,
                        datasets: [{
                            label: 'Number of Visits',
                            data: visits,
                            backgroundColor: 'rgba(0, 123, 255, 0.6)',
                            borderColor: 'rgba(0, 123, 255, 1)',
                            borderWidth: 1
                        }]
                    },
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        scales: {
                            y: {
                                beginAtZero: true,
                                ticks: {
                                    precision: 0
                                }
                            }
                        }
                    }
                });
                
                // Collections Chart
                var collectionsCtx = document.getElementById('collectionsChart').getContext('2d');
                new Chart(collectionsCtx, {
                    type: 'line',
                    data: {
                        labels: formattedDates,
                        datasets: [{
                            label: 'Collection Amount (LKR)',
                            data: collections,
                            fill: false,
                            backgroundColor: 'rgba(40, 167, 69, 0.6)',
                            borderColor: 'rgba(40, 167, 69, 1)',
                            tension: 0.1,
                            pointRadius: 4
                        }]
                    },
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        scales: {
                            y: {
                                beginAtZero: true
                            }
                        }
                    }
                });
            });
        </script>
    <?php elseif ($_SERVER['REQUEST_METHOD'] === 'POST'): ?>
        <div class="alert alert-info text-center">
            <i class="fas fa-info-circle me-2"></i>No data found for the selected criteria.
        </div>
    <?php endif; ?>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
