function getLocation() {
    if (!navigator.geolocation) {
        // Browser doesn't support Geolocation - but don't block user
        console.warn("Geolocation not supported by this browser");
        return;
    }

    // Check if user has blocked location before
    navigator.permissions.query({ name: 'geolocation' }).then(result => {
        if (result.state === 'denied') {
            // User has denied location permanently - allow them to continue
            console.warn("Location permission denied - continuing without GPS");
            return;
        } else {
            // Try to get current location
            navigator.geolocation.getCurrentPosition(fetchLocation, showError, {
                enableHighAccuracy: true,
                timeout: 10000,     // 10 seconds
                maximumAge: 0       // Do not use cached location
            });
        }
    }).catch(err => {
        console.warn("Permission API not supported:", err);
        // Fallback: just try geolocation
        navigator.geolocation.getCurrentPosition(fetchLocation, showError);
    });
}

// ===============================
// Handle successful location
// ===============================
function fetchLocation(position) {
    let latitude = position.coords.latitude;
    let longitude = position.coords.longitude;

    console.log("Latitude:", latitude, "Longitude:", longitude);

    // Check if inside Sri Lanka region
    if (latitude >= 5.9 && latitude <= 9.9 && longitude >= 79.5 && longitude <= 82.0) {
        localStorage.setItem("gpsPermission", "granted");

        // Send to backend
        $.ajax({
            url: "location/location_data.php",
            type: "POST",
            data: { lat: latitude, lon: longitude },
            success: function (response) {
                console.log("Location saved:", response);
            },
            error: function (xhr, status, error) {
                console.error("Error saving location:", error);
            }
        });
    } else {
        console.warn("Location is outside Sri Lanka.");
    }
}

// ===============================
// Handle location errors
// ===============================
function showError(error) {
    let message = "";
    switch (error.code) {
        case error.PERMISSION_DENIED:
            message = "User denied the request for Geolocation.";
            break;
        case error.POSITION_UNAVAILABLE:
            message = "Location information is unavailable.";
            break;
        case error.TIMEOUT:
            message = "The request to get user location timed out.";
            break;
        case error.UNKNOWN_ERROR:
        default:
            message = "An unknown error occurred.";
            break;
    }

    console.warn("Geolocation error:", message);
    
    // Don't redirect - allow user to continue without GPS
    // Location is optional, not mandatory
}

// ===============================
// Run on page load + every 5 mins
// ===============================
window.onload = function () {
    getLocation();

    // Repeat every 5 minutes (300,000 ms)
    setInterval(getLocation, 5 * 60 * 1000);
};