<?php
require_once 'config.php';

// Get vendors for dropdown
$stmt = $pdo->query("SELECT id, NAME as name FROM vendor ORDER BY NAME");
$vendors = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $vendor_id = isset($_POST['vendor_id']) ? $_POST['vendor_id'] : '';
    $latitude  = isset($_POST['latitude']) ? $_POST['latitude'] : '';
    $longitude = isset($_POST['longitude']) ? $_POST['longitude'] : '';
    $address   = isset($_POST['address']) ? $_POST['address'] : '';

    if (!empty($vendor_id) && !empty($latitude) && !empty($longitude)) {
        try {
            $stmt = $pdo->prepare("UPDATE vendor SET latitude = ?, longitude = ?, address = ? WHERE id = ?");
            $stmt->execute([$latitude, $longitude, $address, $vendor_id]);
            
            $success = "Location saved successfully!";
        } catch (PDOException $e) {
            $error = "Error saving location: " . $e->getMessage();
        }
    } else {
        $error = "Please select a vendor and allow location access";
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Vendor Location Mark</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
    <link href="https://cdn.jsdelivr.net/npm/select2-bootstrap-5-theme@1.3.0/dist/select2-bootstrap-5-theme.min.css" rel="stylesheet" />
    <style>
        .container { max-width: 600px; }
        #map { height: 300px; margin: 20px 0; border-radius: 8px; }
        .btn-get-location { margin: 15px 0; }
        .select2-container { width: 100% !important; }
    </style>
</head>
<body>
    <a href="home.php" class="btn btn-light" id="backButton">
            <i class="fas fa-arrow-left me-1 btn btn-primary"></i> Back to Dashbord
    </a>
    <div class="container mt-5">
        <h1 class="text-center mb-4">Vendor Location Mark</h1>
        
        
        <?php if (isset($success)): ?>
            <div class="alert alert-success"><?php echo htmlspecialchars($success, ENT_QUOTES, 'UTF-8'); ?></div>
        <?php endif; ?>
        
        <?php if (isset($error)): ?>
            <div class="alert alert-danger"><?php echo htmlspecialchars($error, ENT_QUOTES, 'UTF-8'); ?></div>
        <?php endif; ?>

        <form method="POST" id="locationForm">
            <div class="mb-3">
                <label for="vendor_id" class="form-label">Search Vendor (by ID or Name):</label>
                <select class="form-select" id="vendor_id" name="vendor_id" required>
                    <option value="">-- Select Vendor --</option>
                    <?php foreach ($vendors as $vendor): ?>
                        <option
                            value="<?php echo htmlspecialchars($vendor['id'], ENT_QUOTES, 'UTF-8'); ?>"
                            data-id="<?php echo htmlspecialchars($vendor['id'], ENT_QUOTES, 'UTF-8'); ?>">
                            <?php echo htmlspecialchars($vendor['name'], ENT_QUOTES, 'UTF-8'); ?>
                            (ID: <?php echo htmlspecialchars($vendor['id'], ENT_QUOTES, 'UTF-8'); ?>)
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>

            <div class="mb-3">
                <button type="button" class="btn btn-primary btn-get-location" onclick="getLocation()">
                    📍 Get Current Location
                </button>
                
                <div id="locationStatus" class="form-text"></div>
                <div id="map"></div>
                
                <input type="hidden" id="latitude" name="latitude">
                <input type="hidden" id="longitude" name="longitude">
                <input type="hidden" id="address" name="address">
            </div>

            <button type="submit" class="btn btn-success w-100">Save Location</button>
        </form>
    </div>

    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>
    
    <script>
        // Initialize Select2
        $(document).ready(function() {
            $('#vendor_id').select2({
                theme: 'bootstrap-5',
                placeholder: 'Search for a vendor by ID or name',
                allowClear: true,
                matcher: function(params, data) {
                    // If there are no search terms, return all of the data
                    if ($.trim(params.term) === '') {
                        return data;
                    }
                    
                    // Search in both the text and the data-id attribute
                    if (data.text.toLowerCase().indexOf(params.term.toLowerCase()) > -1 || 
                        $(data.element).data('id').toString().indexOf(params.term) > -1) {
                        return data;
                    }
                    
                    // Return `null` if the term should not be displayed
                    return null;
                }
            });
        });
        
        let map;
        let marker;
        
        function getLocation() {
            const status = document.getElementById('locationStatus');
            
            if (!navigator.geolocation) {
                status.innerHTML = "Geolocation is not supported by this browser.";
                return;
            }
            
            status.innerHTML = "Locating...";
            
            navigator.geolocation.getCurrentPosition(
                showPosition,
                showError,
                {
                    enableHighAccuracy: true,
                    timeout: 10000,
                    maximumAge: 0
                }
            );
        }
        
        function showPosition(position) {
            const lat = position.coords.latitude;
            const lng = position.coords.longitude;
            
            document.getElementById('latitude').value = lat;
            document.getElementById('longitude').value = lng;
            document.getElementById('locationStatus').innerHTML = 
                `Location found: ${lat.toFixed(6)}, ${lng.toFixed(6)}`;
            
            // Initialize map if not already done
            if (!map) {
                initMap(lat, lng);
            } else {
                map.setView([lat, lng], 15);
                marker.setLatLng([lat, lng]);
            }
            
            // Reverse geocode to get address
            getAddressFromCoordinates(lat, lng);
        }
        
        function initMap(lat, lng) {
            // Using OpenStreetMap with Leaflet (free alternative to Google Maps)
            const leafletScript = document.createElement('script');
            leafletScript.src = 'https://unpkg.com/leaflet@1.7.1/dist/leaflet.js';
            leafletScript.onload = function() {
                const leafletCSS = document.createElement('link');
                leafletCSS.rel = 'stylesheet';
                leafletCSS.href = 'https://unpkg.com/leaflet@1.7.1/dist/leaflet.css';
                document.head.appendChild(leafletCSS);
                
                map = L.map('map').setView([lat, lng], 15);
                
                L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
                    attribution: '© OpenStreetMap contributors'
                }).addTo(map);
                
                marker = L.marker([lat, lng]).addTo(map)
                    .bindPopup('Your location')
                    .openPopup();
            };
            document.head.appendChild(leafletScript);
        }
        
        function getAddressFromCoordinates(lat, lng) {
            // Using OpenStreetMap Nominatim for reverse geocoding
            fetch(`https://nominatim.openstreetmap.org/reverse?format=json&lat=${lat}&lon=${lng}`)
                .then(response => response.json())
                .then(data => {
                    if (data && data.display_name) {
                        document.getElementById('address').value = data.display_name;
                        document.getElementById('locationStatus').innerHTML += 
                            `<br>Address: ${data.display_name}`;
                    }
                })
                .catch(error => {
                    console.error('Error getting address:', error);
                });
        }
        
        function showError(error) {
            const status = document.getElementById('locationStatus');
            switch(error.code) {
                case error.PERMISSION_DENIED:
                    status.innerHTML = "User denied the request for Geolocation.";
                    break;
                case error.POSITION_UNAVAILABLE:
                    status.innerHTML = "Location information is unavailable.";
                    break;
                case error.TIMEOUT:
                    status.innerHTML = "The request to get user location timed out.";
                    break;
                case error.UNKNOWN_ERROR:
                    status.innerHTML = "An unknown error occurred.";
                    break;
            }
        }
        
        // Responsive design adjustments
        function handleResize() {
            if (map) {
                map.invalidateSize();
            }
        }
        
        window.addEventListener('resize', handleResize);
    </script>
</body>
</html>