<?php
session_start();
date_default_timezone_set('Asia/Colombo');
require_once("security/connection_sql.php");

// Basic session check
if (!isset($_SESSION["CURRENT_USER"]) || $_SESSION["CURRENT_USER"] == "") {
    echo "Please Login Again !!!";
    exit();
}

// Ensure PDO throws exceptions
if (isset($conn) && $conn instanceof PDO) {
    $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} else {
    die("Database connection not available.");
}

// Fetch all sales executives for dropdown
$sales_executives = [];
try {
    $stmt = $conn->query("SELECT DISTINCT ref_no1 FROM vendor_visit WHERE ref_no1 IS NOT NULL AND TRIM(ref_no1) <> '' ORDER BY ref_no1");
    $sales_executives = $stmt->fetchAll(PDO::FETCH_COLUMN);
} catch (Throwable $e) {
    $error_message = $e->getMessage();
}

// Handle filters
$selected_executive = isset($_POST['sales_executive']) ? $_POST['sales_executive'] : '';
$date_from = isset($_POST['date_from']) ? $_POST['date_from'] : date('Y-m-d', strtotime('-30 days'));
$date_to = isset($_POST['date_to']) ? $_POST['date_to'] : date('Y-m-d');

// Initialize arrays for data
$main_data = [];
$error_message = null;

// Fetch data if form is submitted
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        // Query to get summary data grouped by collection_note_refno
        $summary_sql = "SELECT 
                collection_note_refno, 
                MIN(sdate1) AS assignment_date, 
                ref_no1 AS sales_executive,
                cus_name,
                SUM(collect) AS total_amount,
                COUNT(*) AS total_items
            FROM vendor_visit 
            WHERE 1=1";

        $params = [];

        // Add filters to query
        if (!empty($selected_executive)) {
            $summary_sql .= " AND ref_no1 = :sales_executive";
            $params[':sales_executive'] = $selected_executive;
        }

        if (!empty($date_from) && !empty($date_to)) {
            $summary_sql .= " AND DATE(sdate1) BETWEEN :date_from AND :date_to";
            $params[':date_from'] = $date_from;
            $params[':date_to'] = $date_to;
        }

        $summary_sql .= " GROUP BY collection_note_refno 
                        ORDER BY assignment_date DESC";


        $stmt = $conn->prepare($summary_sql);
        $stmt->execute($params);
        $main_data = $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Throwable $e) {
        $error_message = $e->getMessage();
    }
}

// Function to fetch details for a specific collection note
function get_collection_details($collection_ref_no) {
    global $conn;
    
    try {
        $sql = "SELECT 
                v.collection_note_refno,
                v.sdate1,
                v.ref_no,
                v.flag,
                v.collect,
                v.cus_name,
                v.latitude,
                v.longitude,
                v.remark,
                v.collection_time
            FROM vendor_visit v
            WHERE v.collection_note_refno = :ref_no
            ORDER BY COALESCE(v.collection_time, v.sdate1) ASC, v.sdate1 ASC";
            
        $stmt = $conn->prepare($sql);
        $stmt->execute([':ref_no' => $collection_ref_no]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Throwable $e) {
        return ['error' => $e->getMessage()];
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Collection View</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        body {
            background-color: #f8f9fa;
        }
        .header-section {
            background: linear-gradient(135deg, #0d6efd 0%, #20c997 100%);
            color: white;
            padding: 2rem 0;
            margin-bottom: 2rem;
        }
        .filter-card {
            border-radius: 10px;
            box-shadow: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
            margin-bottom: 1.5rem;
        }
        .main-table {
            box-shadow: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
        }
        .main-table th {
            background-color: #343a40;
            color: white;
        }
        .main-row {
            cursor: pointer;
            transition: background-color 0.2s;
        }
        .main-row:hover {
            background-color: #f1f8ff;
        }
        .detail-section {
            display: none;
            margin-top: 2rem;
            border-radius: 10px;
            box-shadow: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
            overflow: hidden;
        }
        .detail-header {
            background-color: #20c997;
            color: white;
            padding: 1rem;
        }
        .detail-table th {
            background-color: #495057;
            color: white;
        }
        .no-data-message {
            padding: 2rem;
            text-align: center;
            color: #6c757d;
        }
        .selected-row {
            background-color: #e3f2fd !important;
            font-weight: bold;
        }
    </style>
</head>
<body>
    <div class="header-section">
        <div class="container">
            <div class="row align-items-center">
                <div class="col-md-8">
                    <a href="home.php?url=collection_assign" class="btn btn-light" id="backButton">
                        <i class="fas fa-arrow-left me-1"></i> Back 
                    </a>
            
                    <h1><i class="fas fa-list-check me-2"></i> Collection View</h1>
                   <!-- <p class="lead mb-0">View and track assigned collections</p>-->
                </div>
                <div class="col-md-4 text-end">
                    
                    <a href="collection_assign_report.php" class="btn btn-info ms-2">
                        <i class="fas fa-chart-bar me-1"></i>
                    </a>
                </div>
            </div>
        </div>
    </div>

    <div class="container">
        <!-- Filter Section -->
        <div class="card filter-card">
            <div class="card-header bg-primary text-white">
                <h5 class="mb-0"><i class="fas fa-filter me-2"></i>Filter Collection</h5>
            </div>
            <div class="card-body">
                <form method="POST" class="row g-3">
                    <div class="col-md-4">
                        <label for="sales_executive" class="form-label">Sales Executive</label>
                        <select class="form-select" id="sales_executive" name="sales_executive">
                            <option value="">All Executives</option>
                            <?php foreach ($sales_executives as $exec): ?>
                                <option value="<?= htmlspecialchars($exec) ?>" <?= ($selected_executive === $exec) ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($exec) ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="col-md-3">
                        <label for="date_from" class="form-label">Date From</label>
                        <input type="date" class="form-control" id="date_from" name="date_from" value="<?= htmlspecialchars($date_from) ?>">
                    </div>
                    <div class="col-md-3">
                        <label for="date_to" class="form-label">Date To</label>
                        <input type="date" class="form-control" id="date_to" name="date_to" value="<?= htmlspecialchars($date_to) ?>">
                    </div>
                    <div class="col-md-2 d-flex align-items-end">
                        <button type="submit" class="btn btn-success w-100">
                            <i class="fas fa-search me-1"></i> Search
                        </button>
                    </div>
                </form>
            </div>
        </div>

        <!-- Error Message -->
        <?php if ($error_message): ?>
            <div class="alert alert-danger">
                <i class="fas fa-exclamation-circle me-2"></i>
                <?= htmlspecialchars($error_message) ?>
            </div>
        <?php endif; ?>

        <!-- Main Table -->
        <div class="card main-table mb-4">
            <div class="card-header bg-dark text-white">
                <h5 class="mb-0"><i class="fas fa-table me-2"></i>Collection View</h5>
            </div>
            <div class="card-body p-0">
                <?php if (!empty($main_data)): ?>
                    <div class="table-responsive">
                        <table class="table table-hover mb-0" id="mainTable">
                            <thead>
                                <tr>
                                    <th>Reference No</th>
                                    <th>Date</th>
                                    <th>Sales Executive</th>
                                    <th>Customer</th>
                                    <th class="text-end">Total Amount</th>
                                    <th class="text-center">Items</th>
                                    <th class="text-center">Action</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($main_data as $row): ?>
                                    <tr class="main-row" data-ref="<?= htmlspecialchars($row['collection_note_refno']) ?>">
                                        <td><?= htmlspecialchars($row['collection_note_refno']) ?></td>
                                        <td><?= date('Y-m-d', strtotime($row['assignment_date'])) ?></td>
                                        <td><?= htmlspecialchars($row['sales_executive']) ?></td>
                                        <td><?= htmlspecialchars($row['cus_name']) ?></td>
                                        <td class="text-end"><?= number_format($row['total_amount'], 2) ?></td>
                                        <td class="text-center"><?= $row['total_items'] ?></td>
                                        <td class="text-center">
                                            <button type="button" class="btn btn-sm btn-primary view-details" 
                                                    data-ref="<?= htmlspecialchars($row['collection_note_refno']) ?>">
                                                <i class="fas fa-eye"></i> Details
                                            </button>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php else: ?>
                    <div class="no-data-message">
                        <i class="fas fa-info-circle fa-2x mb-3"></i>
                        <p>No collection data found. Please adjust your filters or create new collection.</p>
                    </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- Detail Section -->
        <div class="detail-section" id="detailSection">
            <div class="detail-header">
                <div class="d-flex justify-content-between align-items-center">
                    <h5 class="mb-0"><i class="fas fa-clipboard-list me-2"></i>collection Details</h5>
                    <button type="button" class="btn-close btn-close-white" aria-label="Close" id="closeDetails"></button>
                </div>
            </div>
            <div class="card-body">
                <div class="row mb-4">
                    <div class="col-md-3">
                        <div class="card h-100">
                            <div class="card-body">
                                <h6 class="card-subtitle mb-2 text-muted">Reference</h6>
                                <p class="card-text fs-5 fw-bold" id="detail-ref-no">-</p>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card h-100">
                            <div class="card-body">
                                <h6 class="card-subtitle mb-2 text-muted">Customer</h6>
                                <p class="card-text fs-5" id="detail-customer">-</p>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card h-100">
                            <div class="card-body">
                                <h6 class="card-subtitle mb-2 text-muted">Date</h6>
                                <p class="card-text fs-5" id="detail-date">-</p>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card h-100">
                            <div class="card-body">
                                <h6 class="card-subtitle mb-2 text-muted">Total Amount</h6>
                                <p class="card-text fs-5 fw-bold text-success" id="detail-amount">-</p>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="table-responsive">
                    <table class="table table-striped table-bordered detail-table">
                        <thead>
                            <tr>
                                <th>Date</th>
                                <th>Type</th>
                                <th>Reference No</th>
                                <th class="text-end">Amount</th>
                                <th>Remarks</th>
                            </tr>
                        </thead>
                        <tbody id="detailTableBody">
                            <!-- Details will be loaded here -->
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    
    <script>
        // Handle clicking on the main table rows
        $(document).ready(function() {
            // View details button click handler
            $('.view-details').click(function() {
                const refNo = $(this).data('ref');
                loadDetails(refNo);
                
                // Highlight the selected row
                $('.main-row').removeClass('selected-row');
                $(this).closest('tr').addClass('selected-row');
            });

            // Close details section
            $('#closeDetails').click(function() {
                $('#detailSection').slideUp();
                $('.main-row').removeClass('selected-row');
            });

            // Function to load details
            function loadDetails(refNo) {
                $.ajax({
                    url: 'collection_assign_details.php',
                    type: 'POST',
                    data: { 
                        action: 'get_details',
                        ref_no: refNo 
                    },
                    dataType: 'json',
                    success: function(response) {
                        if (response.success) {
                            // Display the details section
                            $('#detailSection').slideDown();
                            
                            // Update summary info
                            $('#detail-ref-no').text(refNo);
                            $('#detail-customer').text(response.data[0].cus_name || '-');
                            $('#detail-date').text(response.data[0].sdate1 ? new Date(response.data[0].sdate1).toLocaleDateString() : '-');
                            
                            // Calculate total amount
                            let totalAmount = 0;
                            response.data.forEach(item => {
                                totalAmount += parseFloat(item.collect || 0);
                            });
                            $('#detail-amount').text(new Intl.NumberFormat('en-US', { 
                                style: 'currency', 
                                currency: 'LKR' 
                            }).format(totalAmount));
                            
                            // Clear and populate the detail table
                            let tableContent = '';
                            response.data.forEach(item => {
                                tableContent += `
                                    <tr>
                                        <td>${item.sdate1 ? new Date(item.sdate1).toLocaleDateString() : '-'}</td>
                                        <td>${item.flag === 'checq' ? 'Cheque' : (item.flag === 'inv' ? 'Invoice' : item.flag)}</td>
                                        <td>${item.ref_no || '-'}</td>
                                        <td class="text-end">${parseFloat(item.collect || 0).toFixed(2)}</td>
                                        <td>${item.remark || '-'}</td>
                                    </tr>
                                `;
                            });
                            
                            $('#detailTableBody').html(tableContent);
                        } else {
                            alert('Error loading details: ' + response.message);
                        }
                    },
                    error: function() {
                        alert('Failed to load details. Please try again.');
                    }
                });
            }
        });
    </script>
</body>
</html>
