<?php
// Use the same PDO connection as collection_assign_data_save.php
require_once("security/connection_sql.php");

// Function to get street address from latitude and longitude using OpenStreetMap
function getStreetFromOSM($lat, $lon) {
    if (empty($lat) || empty($lon)) {
        return 'N/A';
    }

    // OSM Nominatim API URL for reverse geocoding
    $url = "https://nominatim.openstreetmap.org/reverse?format=json&lat={$lat}&lon={$lon}";

    // OSM requires a valid User-Agent header
    $options = [
        'http' => [
            'header' => "User-Agent: TApp/1.0\r\n"
        ]
    ];
    $context = stream_context_create($options);

    // Fetch data from the API
    $response = @file_get_contents($url, false, $context);

    if ($response === false) {
        return 'Error fetching address';
    }

    $data = json_decode($response, true);

    // Return the display name if available
    if (isset($data['display_name'])) {
        return $data['display_name'];
    }

    return 'Address not found';
}


// Ensure PDO throws exceptions
if (isset($conn) && $conn instanceof PDO) {
    $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} else {
    $error = "Database connection not available or is not a PDO instance.";
}

// Get distinct users for dropdown filter
$users = [];
if (!isset($error)) {
    try {
        // Get users who have created cheque collections
        $stmt = $conn->query("SELECT DISTINCT created_by FROM cheque_collection WHERE created_by IS NOT NULL AND created_by != '' ORDER BY created_by");
        $users = $stmt->fetchAll(PDO::FETCH_COLUMN);
    } catch(PDOException $e) {
        $error = "Error fetching users: " . $e->getMessage();
    }
}

// Process form submission
$report_data = [];
$summary = [];
$filters = [];
$total_amount = 0;
$total_cheques = 0;

if ($_SERVER['REQUEST_METHOD'] === 'POST' && !isset($error)) {
    $created_by = isset($_POST['created_by']) ? $_POST['created_by'] : '';
    $start_date = isset($_POST['start_date']) ? $_POST['start_date'] : '';
    $end_date = isset($_POST['end_date']) ? $_POST['end_date'] : '';
    
    $filters = [
        'created_by' => $created_by,
        'start_date' => $start_date,
        'end_date' => $end_date
    ];
    
    try {
        // Build query for cheque_collection table
        $sql = "SELECT cc.date, cc.cheque_no, cc.cus_name, cc.amount, cc.created_by, cc.latitude, cc.longitude 
                FROM cheque_collection cc
                WHERE cc.date BETWEEN :start_date AND :end_date";
        
        $params = [
            ':start_date' => $start_date,
            ':end_date' => $end_date
        ];
        
        if (!empty($created_by)) {
            $sql .= " AND cc.created_by = :created_by";
            $params[':created_by'] = $created_by;
        }
        
        $sql .= " ORDER BY cc.date DESC, cc.created_by";
        
        $stmt = $conn->prepare($sql);
        $stmt->execute($params);
        $report_data = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Calculate summary
        if ($report_data) {
            $total_cheques = count($report_data);
            foreach ($report_data as $record) {
                $total_amount += $record['amount'];
            }
        }
        
    } catch(PDOException $e) {
        $error = "Error generating report: " . $e->getMessage();
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Cheque Collection Report</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        body { background-color: #f8f9fa; }
        .report-header {
            background: linear-gradient(135deg, #17a2b8 0%, #28a745 100%);
            color: white;
            padding: 2rem 0;
            margin-bottom: 2rem;
        }
        .summary-card {
            border: none;
            border-radius: 15px;
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
            transition: transform 0.3s ease;
        }
        .summary-card:hover { transform: translateY(-5px); }
        .report-table { box-shadow: 0 0 20px rgba(0,0,0,0.1); }
        @media print {
            .no-print { display: none; }
            .summary-card { box-shadow: none; border: 1px solid #ddd; }
            body { background-color: #fff; }
        }
    </style>
</head>
<body>
    <div class="report-header">
        <div class="container">
            <div class="row align-items-center">
                <div class="col-md-8">
                    <h1><i class="fas fa-money-check-alt me-2"></i>Cheque Collection Report</h1>
                    <p class="lead mb-0">Track and analyze cheque collections</p>
                </div>
                <div class="col-md-4 text-end">
                    <button onclick="window.print()" class="btn btn-light btn-sm no-print">
                        <i class="fas fa-print me-1"></i>Print Report
                    </button>
                </div>
            </div>
        </div>
    </div>

    <div class="container">
        <!-- Filter Form -->
        <div class="card mb-4 no-print">
            <div class="card-header bg-primary text-white">
                <h5 class="mb-0"><i class="fas fa-filter me-2"></i>Filter Report</h5>
            </div>
            <div class="card-body">
                <form method="POST" class="row g-3">
                    <div class="col-md-4">
                        <label for="created_by" class="form-label">Select User</label>
                        <select class="form-select" id="created_by" name="created_by">
                            <option value="">All Users</option>
                            <?php foreach ($users as $user): ?>
                                <option value="<?= htmlspecialchars($user) ?>" 
                                    <?= (isset($filters['created_by']) && $filters['created_by'] == $user) ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($user) ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="col-md-3">
                        <label for="start_date" class="form-label">Start Date</label>
                        <input type="date" class="form-control" id="start_date" name="start_date" 
                               value="<?= isset($filters['start_date']) ? $filters['start_date'] : '' ?>" required>
                    </div>
                    <div class="col-md-3">
                        <label for="end_date" class="form-label">End Date</label>
                        <input type="date" class="form-control" id="end_date" name="end_date" 
                               value="<?= isset($filters['end_date']) ? $filters['end_date'] : '' ?>" required>
                    </div>
                    <div class="col-md-2 d-flex align-items-end">
                        <button type="submit" class="btn btn-success w-100">
                            <i class="fas fa-chart-bar me-1"></i>Generate
                        </button>
                    </div>
                </form>
            </div>
        </div>

        <?php if (isset($error)): ?>
            <div class="alert alert-danger"><?= $error ?></div>
        <?php endif; ?>

        <?php if ($_SERVER['REQUEST_METHOD'] === 'POST' && empty($error)): ?>
            <?php if (!empty($report_data)): ?>
                <!-- Summary Cards -->
                <div class="row mb-4">
                    <div class="col-md-6 mb-3">
                        <div class="card summary-card text-center bg-light">
                            <div class="card-body">
                                <i class="fas fa-coins fa-2x text-success mb-2"></i>
                                <h4 class="card-title">LKR <?= number_format($total_amount, 2) ?></h4>
                                <p class="card-text text-muted">Total Collection Amount</p>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-6 mb-3">
                        <div class="card summary-card text-center bg-light">
                            <div class="card-body">
                                <i class="fas fa-file-invoice-dollar fa-2x text-info mb-2"></i>
                                <h4 class="card-title"><?= $total_cheques ?></h4>
                                <p class="card-text text-muted">Total Cheques Collected</p>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Detailed Report -->
                <div class="card">
                    <div class="card-header bg-success text-white">
                        <h5 class="mb-0"><i class="fas fa-list-alt me-2"></i>Detailed Cheque Collection Report</h5>
                    </div>
                    <div class="card-body p-0">
                        <div class="table-responsive">
                            <table class="table table-striped table-hover report-table mb-0">
                                <thead class="table-dark">
                                    <tr>
                                        <th>Date</th>
                                        <th>Cheque No</th>
                                        <th>Customer</th>
                                        <th>Collected By</th>
                                        <th>Location</th>
                                        <th class="text-end">Cheque Amount</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($report_data as $record): ?>
                                        <tr>
                                            <td><?= date('M j, Y', strtotime($record['date'])) ?></td>
                                            <td><?= htmlspecialchars($record['cheque_no']) ?></td>
                                            <td><?= htmlspecialchars($record['cus_name']) ?></td>
                                            <td><?= htmlspecialchars(isset($record['created_by']) ? $record['created_by'] : 'N/A') ?></td>
                                            <td><?= htmlspecialchars(getStreetFromOSM($record['latitude'], $record['longitude'])) ?></td>
                                            <td class="text-end"><?= number_format($record['amount'], 2) ?></td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                                <tfoot class="table-light">
                                    <tr>
                                        <th colspan="5" class="text-end">Total:</th>
                                        <th class="text-end">LKR <?= number_format($total_amount, 2) ?></th>
                                    </tr>
                                </tfoot>
                            </table>
                        </div>
                    </div>
                </div>

            <?php else: ?>
                <div class="alert alert-info text-center">
                    <i class="fas fa-info-circle me-2"></i>No cheque collection records found for the selected criteria.
                </div>
            <?php endif; ?>
        <?php endif; ?>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Set default dates to current month
        document.addEventListener('DOMContentLoaded', function() {
            const startDateInput = document.getElementById('start_date');
            const endDateInput = document.getElementById('end_date');
            
            if (!startDateInput.value || !endDateInput.value) {
                const today = new Date();
                const firstDay = new Date(today.getFullYear(), today.getMonth(), 1);
                
                startDateInput.value = firstDay.toISOString().split('T')[0];
                endDateInput.value = today.toISOString().split('T')[0];
            }
        });
    </script>
</body>
</html>

